
import java.io.*;
import java.sql.*;
import java.util.*;

// This is a singleton class.
public class DBConnectionPool implements Runnable
{
   // Create the only instance, make it private static.
   private static DBConnectionPool m_Instance = new DBConnectionPool();

   private String m_Driver;
   private String m_URL;
   private String m_UserName;
   private String m_Password;
   private Integer m_InitialConnections;
   private Integer m_MaxConnections;
   private Vector m_AvailableConnections;
   private Vector m_BusyConnections;
   private boolean m_ConnectionPending = false;

   // Protect the constructor, so no other class can call it.
   // This is called once when the class is first loaded.
   private DBConnectionPool()
   {
      if ( readConnectInfo() )
      {
         m_AvailableConnections = new Vector( m_InitialConnections.intValue() );
         m_BusyConnections = new Vector();

         if ( createInitialConnections() )
         {
            System.out.println("DBConnectionPool constructed.");
         }
      }
   }

   // Make the static instance publicly available.
   public static DBConnectionPool getInstance()
   {
      return m_Instance;
   }

   /**
   * Returns the number of initial connections specified in the
   * database connect configuration file.
   */
   public synchronized Integer getInitialConnections()
   {
      return m_InitialConnections;
   }

   public synchronized Integer getMaxConnections()
   {
      return m_MaxConnections;
   }

   /**
   * This property is useful for bumping the max connections
   * dynamically from a servlet, for example, to increase throughput.
   */
   public synchronized void setMaxConnections( Integer MaxConnections )
   {
      m_MaxConnections = MaxConnections;
   }

   /**
   * Return the number of currently available connections.
   * This is useful for reporting in a servlet.
   */
   public synchronized Integer getAvailableConnections()
   {
      return new Integer( m_AvailableConnections.size() );
   }

   /**
   * Return the number of currently busy connections.
   * This is useful for reporting in a servlet.
   */
   public synchronized Integer getBusyConnections()
   {
      return new Integer( m_BusyConnections.size() );
   }

   public synchronized String toString()
   {
      StringBuffer State = new StringBuffer();

      State.append( "ConnectionPool" );
      State.append( "\nURL \t" + m_URL );
      State.append( "\nDriver \t" + m_Driver );
      State.append( "\nInitial Connections \t" + m_InitialConnections );
      State.append( "\nMax Connections \t" + m_MaxConnections );
      State.append( "\nAvailable Connections \t" + m_AvailableConnections.size() );
      State.append( "\nBusy Connections \t" + m_BusyConnections.size() );

      return State.toString();
   }

   public synchronized Connection getConnection()
   {
      boolean connectionClosed = true;

      if (m_AvailableConnections.isEmpty() == false)
      {
         // Pull off available connection.
         Connection existingConnection = (Connection)m_AvailableConnections.lastElement();
         int lastIndex = m_AvailableConnections.size() - 1;
         m_AvailableConnections.removeElementAt( lastIndex );

         // If existing connection was closed (e.g., it timed out),
         // remove it from available, repeat process of getting connection,
         // notify any threads waiting because the connection limit was reached.
         try
         {
            connectionClosed = existingConnection.isClosed();
         }
         catch ( SQLException exception )
         {
            connectionClosed = true;
         }

         if ( connectionClosed )
         {
            notifyAll();
            return ( getConnection() );
         }
         else
         {
            m_BusyConnections.addElement( existingConnection );
            return ( existingConnection );
         }
      }
      else
      {
         if ( (totalConnections() < m_MaxConnections.intValue()) &&
              (m_ConnectionPending == false) )
         {
            makeBackgroundConnection();
         }

         try
         {
            wait();
         }
         catch( InterruptedException exception )
         {
            // Ignore.
         }

         // Someone freed up a connection, so try again.
         return ( getConnection() );
      }
   }

   private boolean createInitialConnections()
   {
      for (int i = 0; i < m_InitialConnections.intValue(); i++)
      {
         Connection connection = newConnection();

         if (connection != null)
         {
            m_AvailableConnections.addElement( connection );
         }
         else
         {
            return false;
         }
      }

      return true;
   }

   private Connection newConnection()
   {
      try
      {
         // Load JDBC driver.
         Class.forName(m_Driver);

         // Obtain DB connection.
         Connection dbConnection = DriverManager.getConnection(m_URL, m_UserName, m_Password);
         dbConnection.setAutoCommit(false);

         return dbConnection;
      }
      // Thrown by Class.forName
      catch (ClassNotFoundException Exception)
      {
         System.out.println("Can't find class for JDBC driver: " + m_Driver);
         System.out.println(Exception);
         return null;
      }
      // Thrown by DriverManager.getConnection
      catch (SQLException Exception)
      {
         System.out.println("Unable to connect to the database: " + m_URL);
         System.out.println(Exception);
         return null;
      }
   }

   private void makeBackgroundConnection()
   {
      m_ConnectionPending = true;

      try
      {
         Thread connectThread = new Thread(this);
         connectThread.start();
      }
      catch (OutOfMemoryError error)
      {
         // Dummy. Log this in the future.
      }
   }

   public void run()
   {
      try
      {
         Connection connection = newConnection();
         synchronized(this)
         {
            m_AvailableConnections.addElement(connection);
            m_ConnectionPending = false;
            notifyAll();
         }
      }
      catch (Exception exception)
      {
         // Dummy. Log this in the future.
         // Give up and wait for existing one to free up.
      }
   }

   public synchronized int totalConnections()
   {
      return ( m_AvailableConnections.size() + m_BusyConnections.size() );
   }

   /**
   * Clients should call this method to return connections
   * back to the connection pool when they are done with them.
   */
   public synchronized void free(Connection connection)
   {
      m_BusyConnections.removeElement( connection );
      m_AvailableConnections.addElement( connection );

      // Wake up threads waiting for a connection.
      notifyAll();
   }

   /**
   * Closes all connections. <b>This method is dangerous</b>.
   * Be sure no connections are in use before calling.
   * Note that you are not <i>required</i> to call this when
   * done, since connections are guaranteed to be closed when
   * garabage collected.
   */
   public synchronized void closeAllConnections()
   {
      closeConnections( m_AvailableConnections );
      m_AvailableConnections = new Vector();

      closeConnections( m_BusyConnections );
      m_BusyConnections = new Vector();
   }

   private void closeConnections(Vector connections)
   {
      try
      {
         for (int i = 0; i < connections.size(); i++)
         {
            Connection connection = (Connection)connections.elementAt(i);

            if ( !connection.isClosed() )
            {
               connection.close();
            }
         }
      }
      catch (SQLException exception)
      {
         // Ignored.
      }
   }

   private boolean readConnectInfo()
   {
      final String PROPERTIES_FILENAME = "database.properties";
      final String PROPERTY_URL = "url";
      final String PROPERTY_DRIVER = "driver";
      final String PROPERTY_USERNAME = "username";
      final String PROPERTY_PASSWORD = "password";
      final String PROPERTY_INITIAL_CONNECTIONS = "initialconnections";
      final String PROPERTY_MAX_CONNECTIONS = "maxconnections";

      FileInputStream propertiesFile = null;

      try
      {
         boolean propertyNotFound = false;

         propertiesFile = new FileInputStream( PROPERTIES_FILENAME );

         Properties dbProperties = new Properties();
         dbProperties.load( propertiesFile );

         m_Driver = dbProperties.getProperty( PROPERTY_DRIVER );
         if ( m_Driver == null )
         {
            System.out.println("Property " + PROPERTY_DRIVER + " not found in " + PROPERTIES_FILENAME);
            propertyNotFound = true;
         }

         m_URL = dbProperties.getProperty( PROPERTY_URL );
         if ( m_URL == null )
         {
            System.out.println("Property " + PROPERTY_URL + " not found in " + PROPERTIES_FILENAME);
            propertyNotFound = true;
         }

         m_UserName = dbProperties.getProperty( PROPERTY_USERNAME );
         if ( m_UserName == null )
         {
            System.out.println("Property " + PROPERTY_USERNAME + " not found in " + PROPERTIES_FILENAME);
            propertyNotFound = true;
         }

         m_Password = dbProperties.getProperty( PROPERTY_PASSWORD );
         if ( m_Password == null )
         {
            System.out.println("Property " + PROPERTY_PASSWORD + " not found in " + PROPERTIES_FILENAME);
            propertyNotFound = true;
         }

         String InitialConnections = dbProperties.getProperty( PROPERTY_INITIAL_CONNECTIONS );
         if ( InitialConnections == null )
         {
            System.out.println("Property " + PROPERTY_INITIAL_CONNECTIONS + " not found in " + PROPERTIES_FILENAME);
            propertyNotFound = true;
         }
         else
         {
            m_InitialConnections = new Integer( InitialConnections );
         }

         String MaxConnections = dbProperties.getProperty( PROPERTY_MAX_CONNECTIONS );
         if ( MaxConnections == null )
         {
            System.out.println("Property " + PROPERTY_MAX_CONNECTIONS + " not found in " + PROPERTIES_FILENAME);
            propertyNotFound = true;
         }
         else
         {
            m_MaxConnections = new Integer( MaxConnections );
         }

         if ( m_InitialConnections.intValue() > m_MaxConnections.intValue() )
         {
            m_InitialConnections = m_MaxConnections;
         }

         if ( propertyNotFound )
            return false;
         else
            return true;
      }
      catch( FileNotFoundException exception )
      {
         System.out.println("The properties file with database connect information was not found.");
         System.out.println( exception );
         return false;
      }
      catch( Exception exception )
      {
         System.out.println("Unknown error attempting to read the properties file with database connect information.");
         return false;
      }
      finally
      {
         try
         {
            propertiesFile.close();
         }
         catch ( Exception ignored )
         {
         }
      }
   }
}


